﻿using Moq;
using NUnit.Framework;
using NVCC.Models;
using NVCC.WebUI.Controllers;
using NVCC.WebUI.Infrastructure;
using NVCC.WebUI.Models;
using System.Linq;
using System.Web;
using System.Web.Mvc;

namespace NVCC.WebUI.Tests.Controllers
{
    [TestFixture]
    public class HomeControllerTests
    {
        private Mock<IPatientProfileService> _patientRepoMock;
        private Mock<IUserService> _userServiceMock;
        private HomeController _controller;

        [SetUp]
        public void TestSetUp()
        {
            _patientRepoMock = new Mock<IPatientProfileService>();
            _userServiceMock = new Mock<IUserService>();

            _userServiceMock.Setup(m => m.GetViaUser(It.IsAny<string>(), It.IsAny<string>(), It.IsAny<string>(), It.IsAny<short>()))
                            .Returns(FakeDataManager.GetViaUser(viaBApproved: true));


            _controller = new HomeController(_patientRepoMock.Object,
                                            _userServiceMock.Object);
        }

        [Test]
        public void IndexActionGet_ExistingUserValidAccess_ReturnsViewResult()
        {
            // Arrange
            _userServiceMock.Setup(m => m.GetUser()).Returns(FakeDataManager.GetUser(0));

            // Act  
            var result = _controller.Index() as ViewResult;

            // Assert
            Assert.IsInstanceOf(typeof(ViewResult), result);
            Assert.IsNotNull(result);
            Assert.IsInstanceOf(typeof(HomePageViewModel), result.Model);
        }

        [Test]
        public void IndexActionGet_NoActiveSessionButHasFacility_RedirectsToLogin()
        {
            // Arrange
            _userServiceMock.Setup(m => m.GetUser()).Returns(FakeDataManager.GetUser(2));

            // Act
            var result = _controller.Index();

            // Assert
            Assert.IsNotNull(result);
            Assert.IsInstanceOf(typeof(RedirectToRouteResult), result);
            Assert.AreEqual("Login", ((RedirectToRouteResult)result).RouteValues["action"].ToString());
            Assert.AreEqual("User", ((RedirectToRouteResult)result).RouteValues["controller"].ToString());
        }

        [Test]
        public void IndexAction_HttpViaUserViaBFalse_RedirectsToLogin()
        {
            // Arrange
            _userServiceMock.Setup(m => m.GetUser()).Returns(FakeDataManager.GetUser(3));

            // Act
            var result = _controller.Index();
            // Assert
            Assert.IsNotNull(result);
            Assert.IsInstanceOf(typeof(RedirectToRouteResult), result);
            Assert.AreEqual("Login", ((RedirectToRouteResult)result).RouteValues["action"].ToString());
            Assert.AreEqual("User", ((RedirectToRouteResult)result).RouteValues["controller"].ToString());
        }

        [Test]
        public void IndexAction_HttpUserProfilePhiPiiApprovedFalse_RedirectsToLogin()
        {
            // Arrange
            _userServiceMock.Setup(m => m.GetUser()).Returns(FakeDataManager.GetUser(4));
            // Act
            var result = _controller.Index();
            // Assert
            Assert.IsNotNull(result);
            Assert.IsInstanceOf(typeof(RedirectToRouteResult), result);
            Assert.AreEqual("Login", ((RedirectToRouteResult)result).RouteValues["action"].ToString());
            Assert.AreEqual("User", ((RedirectToRouteResult)result).RouteValues["controller"].ToString());
        }

        [Test]
        public void IndexActionPost_PatientSsnIsValid_RedirectsToPatientProfile()
        {
            // Arrange
            _userServiceMock.Setup(m => m.GetUser()).Returns(FakeDataManager.GetUser());
            var model = new HomePageViewModel { PatientSSN = "999999999", sta3n = (short)888 };
            _patientRepoMock.Setup(m => m.GetPatient(It.IsAny<string>(), It.IsAny<short>()))
                            .Returns(FakeDataManager.GetPatient());
            HttpContextManager.SetCurrentContext(FakeHttpContext.GetMockedHttpContext());

            // Act
            var result = _controller.Index(model);
            // Assert
            Assert.IsNotNull(result);
            Assert.IsInstanceOf(typeof(RedirectToRouteResult), result);
            Assert.AreEqual("Index", ((RedirectToRouteResult)result).RouteValues["action"].ToString());
            Assert.AreEqual("PatientProfile", ((RedirectToRouteResult)result).RouteValues["controller"].ToString());
        }

        [Test]
        public void IndexActionPost_PatientSsnIsNull_ReturnsViewResult()
        {
            // Arrange
            _userServiceMock.Setup(m => m.GetUser()).Returns(FakeDataManager.GetUser());
            var model = new HomePageViewModel { PatientSSN = null, sta3n = 888 };

            // Act
            var result = _controller.Index(model);

            // Assert
            Assert.IsNotNull(result);
            Assert.IsInstanceOf(typeof(ViewResult), result);
            Assert.IsInstanceOf(typeof(HomePageViewModel), ((ViewResult)result).Model);
            Assert.IsFalse(((ViewResult)result).ViewData.ModelState.IsValid);
            ModelState ms;
            Assert.IsTrue(((ViewResult)result).ViewData.ModelState.TryGetValue("PatientSSN", out ms));
            Assert.IsTrue(ms.Errors.Count > 0);
        }

        [Test]
        public void IndexActionPost_PatientSsnLessThan10Numbers_ReturnsViewResult()
        {
            // Arrange
            _userServiceMock.Setup(m => m.GetUser()).Returns(FakeDataManager.GetUser());
            HomePageViewModel model = new HomePageViewModel { PatientSSN = "555", sta3n = 888 };

            // Act
            var result = _controller.Index(model);

            Assert.IsNotNull(result);
            Assert.IsInstanceOf(typeof(ViewResult), result);
            Assert.IsInstanceOf(typeof(HomePageViewModel), ((ViewResult)result).Model);
            Assert.IsFalse(((ViewResult)result).ViewData.ModelState.IsValid);
            ModelState ms;
            Assert.IsTrue(((ViewResult)result).ViewData.ModelState.TryGetValue("PatientSSN", out ms));
            Assert.IsTrue(ms.Errors.Count > 0);
        }

        [Test]
        public void IndexActionPost_PatientSSNNotAllNumbers_ReturnsErrorModelState()
        {
            //Arrange
            _userServiceMock.Setup(m => m.GetUser()).Returns(FakeDataManager.GetUser());
            HomePageViewModel model = new HomePageViewModel { PatientSSN = "12345678A" };

            // Act
            var result = _controller.Index(model);

            Assert.IsNotNull(result);
            Assert.IsInstanceOf(typeof(ViewResult), result);
            Assert.IsFalse(((ViewResult)result).ViewData.ModelState.IsValid);
            ModelState ms;
            Assert.IsTrue(((ViewResult)result).ViewData.ModelState.TryGetValue("PatientSSN", out ms));
            Assert.IsNotNull(ms);
            Assert.IsNotNull(ms.Errors);
            Assert.IsTrue(ms.Errors.Count > 0);
            var theError = ms.Errors.FirstOrDefault();
            Assert.IsNotNull(theError);

            var errorMessage = theError.ErrorMessage;
            Assert.AreEqual("SSN must be only numbers, spaces, and dashes", errorMessage);
        }

        [Test]
        public void IndexActionPost_PatientNotFound_ReturnsViewResult()
        {
            // Arrange
            _userServiceMock.Setup(m => m.GetUser()).Returns(FakeDataManager.GetUser());
            var model = new HomePageViewModel { PatientSSN = "999999999", sta3n = 888 };
            _patientRepoMock.Setup(m => m.GetPatient(It.IsAny<string>(), It.IsAny<short>())).Returns((Patient)null);

            // Act
            var result = _controller.Index(model);

            Assert.IsNotNull(result);
            Assert.IsInstanceOf(typeof(ViewResult), result);
            Assert.IsInstanceOf(typeof(HomePageViewModel), ((ViewResult)result).Model);
            Assert.IsFalse(((ViewResult)result).ViewData.ModelState.IsValid);
            ModelState ms;
            Assert.IsTrue(((ViewResult)result).ViewData.ModelState.TryGetValue("PatientSSN", out ms));
            Assert.IsTrue(ms.Errors.Count > 0);
        }

        [Test]
        public void IndexActionPost_PatientSta3n0_ReturnsErrorModelState()
        {
            //Arrange
            _userServiceMock.Setup(m => m.GetUser()).Returns(FakeDataManager.GetUser());
            HomePageViewModel model = new HomePageViewModel { PatientSSN = "123456789", sta3n = 0 };

            // Act
            var result = _controller.Index(model);

            Assert.IsNotNull(result);
            Assert.IsInstanceOf(typeof(ViewResult), result);
            Assert.IsFalse(((ViewResult)result).ViewData.ModelState.IsValid);
            ModelState ms;
            Assert.IsTrue(((ViewResult)result).ViewData.ModelState.TryGetValue("sta3n", out ms));
            Assert.IsNotNull(ms);
            Assert.IsNotNull(ms.Errors);
            Assert.IsTrue(ms.Errors.Count > 0);
            var theError = ms.Errors.FirstOrDefault();
            Assert.IsNotNull(theError);

            var errorMessage = theError.ErrorMessage;
            Assert.AreEqual("Station selected is not active", errorMessage);
        }

        [Test]
        public void UnauthorizedAction_ReturnsEmptyview()
        {
            //Arrange
            _patientRepoMock.Setup(m => m.GetPatient(It.IsAny<string>(), It.IsAny<short>())).Returns((Patient)null);

            // Act
            var result = _controller.Unauthorized();

            Assert.IsNotNull(result);
            Assert.IsInstanceOf(typeof(ViewResult), result);
        }
    }
}
